import {
	Plane,
	Program,
	Mat4,
	Mesh,
	Renderer,
	Texture,
	Vec4,
} from 'https://unpkg.com/ogl@0.0.49/src/index.mjs';

import { transpose } from 'https://unpkg.com/ogl@0.0.49/src/math/functions/Mat4Func.js';

class ColorMatrix {
	constructor() {
		this._columnCount = 5;
		this._rowCount = 4;
		this._cellCount = this._columnCount * this._rowCount;
		
		this._matrix = new Array(this._cellCount).fill(0).map((_, index) => {
			const row = Math.floor(index / this._columnCount);
			
			return index % this._columnCount === row ? 1 : 0;
		});
	}
	
	getOffsetVector() {
		return this._matrix.filter((_, index) => index % this._columnCount === 4);
	}
	
	getRgbaMatrix() {
		return this._matrix.filter((_, index) => {
			const column = index % this._columnCount;
			
			return column < 4;
		});
	}
	
	set(...args) {
		if (args.length !== this._cellCount) {
			throw new Error(`Number of values must be ${this._cellCount}`);
		}
		
		this._matrix = args;
		
		return this;
	}
}

class ImageLoader {
	static loadFromUrl(src) {
		const image = new Image();
		
		image.setAttribute('anonymous', true);
		image.setAttribute('crossorigin', true);
		
		return new Promise((resolve, reject) => {
			image.addEventListener('load', () => resolve(image));
			image.addEventListener('error', () => reject(new Error('Unable to load image')));
			image.src = src;
		});
	}
	
	static loadFromFile(file) {
		return ImageLoader.loadFromUrl(URL.createObjectURL(file));
	}
}

class Vec4Utils {
	static multiply(a, b) {
		return new Vec4(new Array(4).fill(1).map((_, index) => {
			return a[index] * b[index];
		}));
	}
}

class Application {
	constructor() {
		this._renderer = new Renderer();
		this._map = new Texture(this._gl);
		this._mesh = this.createMesh(this._map);
	}
	
	get canvas() {
		return this._renderer.gl.canvas;
	}
	
	createMesh(map) {
		const geometry = new Plane(this._gl, {
			height: 2,
			width: 2,
		});
		
		const program = new Program(this._gl, {
			fragment: `
				precision mediump float;

				uniform sampler2D u_map;

				uniform mat4 u_brightnessMatrix;
				uniform vec4 u_brightnessOffset;
				uniform mat4 u_contrastMatrix;
				uniform vec4 u_contrastOffset;
				uniform mat4 u_filterMatrix;
				uniform vec4 u_filterOffset;
				uniform mat4 u_exposureMatrix;
				uniform vec4 u_exposureOffset;
				uniform mat4 u_saturationMatrix;
				uniform vec4 u_saturationOffset;

				varying vec2 v_uv;

				void main() {
					vec4 texel = texture2D(u_map, v_uv);
					mat4 matrix = u_brightnessMatrix * u_contrastMatrix * u_exposureMatrix * u_saturationMatrix * u_filterMatrix;
					vec4 offset = u_brightnessOffset + u_contrastOffset + u_exposureOffset + u_saturationOffset * u_filterOffset;

					// Diff view
					// gl_FragColor = mix(texel, matrix * texel + offset, step(0.5, v_uv.x + 0.75 * (-v_uv.y + 0.5)));
					gl_FragColor = matrix * texel + offset;
				}
			`,
			vertex: `
				attribute vec4 position;
				attribute vec2 uv;

				varying vec2 v_uv;

				void main() {
					v_uv = uv;

					gl_Position = position;
				}
			`,
			
			uniforms: {
				u_brightnessMatrix: { value: new Mat4() },
				u_brightnessOffset: { value: new Vec4() },
				u_contrastMatrix: { value: new Mat4() },
				u_contrastOffset: { value: new Vec4() },
				u_filterMatrix: { value: new Mat4() },
				u_filterOffset: { value: new Vec4() },
				u_exposureMatrix: { value: new Mat4() },
				u_exposureOffset: { value: new Vec4() },
				u_saturationMatrix: { value: new Mat4() },
				u_saturationOffset: { value: new Vec4() },
				u_map: { value: map },
			},
		});
		
		return new Mesh(this._gl, {
			geometry,
			program,
		});
	}
	
	setBrightness(brightness) {
		this._mesh.program.uniforms.u_brightnessOffset.value.set(
			brightness,
			brightness,
			brightness,
			0,
		);
		
		this.render();
	}
	
	setContrast(contrast) {
		const c = 1 + contrast;
		const o = 0.5 * (1 - c);
		
		transpose(this._mesh.program.uniforms.u_contrastMatrix.value, [
			c, 0, 0, 0,
			0, c, 0, 0,
			0, 0, c, 0,
			0, 0, 0, 1,
		]);
		
		this._mesh.program.uniforms.u_contrastOffset.value.set(o, o, o, 0);
		
		this.render();
	}
	
	setExposure(exposure) {
		const e = 1 + exposure;
		
		transpose(this._mesh.program.uniforms.u_exposureMatrix.value, [
			e, 0, 0, 0,
			0, e, 0, 0,
			0, 0, e, 0,
			0, 0, 0, 1,
		]);
		
		this.render();
	}
	
	setFilter(filterName) {
		const colorMatrix = Application._filterMatrixMap.get(filterName);
		const matrix = colorMatrix.getRgbaMatrix();
		const offset = colorMatrix.getOffsetVector();
		
		transpose(this._mesh.program.uniforms.u_filterMatrix.value, matrix);
		
		this._mesh.program.uniforms.u_filterOffset.value.set(...offset);
		
		this.render();
	}
	
	setImage(image) {
		this._map.image = image;
		this._map.needsUpdate = true;
		
		this.setSize(image.naturalWidth, image.naturalHeight);
		this.render();
	}
	
	setSaturation(saturation) {
		const s = 1 + saturation;
		
		// https://www.w3.org/TR/WCAG20/#relativeluminancedef
		const lr = 0.2126;
		const lg = 0.7152;
		const lb = 0.0722;
		
		const sr = (1 - s) * lr;
		const sg = (1 - s) * lg;
		const sb = (1 - s) * lb;
		
		this._mesh.program.uniforms.u_saturationMatrix.value.set(
			sr + s, sr    , sr    , 0,
			sg    , sg + s, sg    , 0,
			sb    , sb    , sb + s, 0,
			0     , 0     , 0     , 1,
		);
		
		this.render();
	}
	
	setSize(width, height) {
		this._renderer.setSize(width, height);
	}
	
	render() {
		this._renderer.render({
			scene: this._mesh,
		});
	}
	
	get _gl() {
		return this._renderer.gl;
	}
	
	static get _filterMatrixMap() {
		return new Map([
			['none', new ColorMatrix().set(
				1, 0, 0, 0, 0,
				0, 1, 0, 0, 0,
				0, 0, 1, 0, 0,
				0, 0, 0, 1, 0,
			)],
			['chrome', new ColorMatrix().set(
				 1.398, -0.316,  0.065, -0.273, 0.201,
				-0.051,  1.278, -0.080, -0.273, 0.201,
				-0.051,  0.119,  1.151, -0.290, 0.215,
				 0.000,  0.000,  0.000,  1.000, 0.000,
			)],
			['fade', new ColorMatrix().set(
				1.073, -0.015, 0.092, -0.115, -0.017,
				0.107,  0.859, 0.184, -0.115, -0.017,
				0.015,  0.077, 1.104, -0.115, -0.017,
				0.000,  0.000, 0.000,  1.000,  0.000,
			)],
			['monochrome', new ColorMatrix().set(
				0.212, 0.715, 0.114, 0.000, 0.000,
				0.212, 0.715, 0.114, 0.000, 0.000,
				0.212, 0.715, 0.114, 0.000, 0.000,
				0.000, 0.000, 0.000, 1.000, 0.000
			)],
			['noir', new ColorMatrix().set(
				0.150, 1.300, -0.250, 0.100, -0.200,
				0.150, 1.300, -0.250, 0.100, -0.200,
				0.150, 1.300, -0.250, 0.100, -0.200,
				0.000, 0.000,  0.000, 1.000,  0.000
			)],
			['rgb2gbr', new ColorMatrix().set(
				0, 1, 0, 0, 0,
				0, 0, 1, 0, 0,
				1, 0, 0, 0, 0,
				0, 0, 0, 1, 0,
			)],
			['summer', new ColorMatrix().set(
				0.95, 0.20, 0.00, 0.00, 0.00,
				0.10, 0.90, 0.00, 0.00, 0.00,
				0.00, 0.00, 0.90, 0.00, 0.00,
				0.00, 0.00, 0.00, 1.00, 0.00,
			)],
		]);
	}
}

(async () => {
	const inputBrightness = document.getElementById('input-brightness');
	const inputContrast = document.getElementById('input-contrast');
	const inputExposure = document.getElementById('input-exposure');
	const inputFilterList = Array.from(document.querySelectorAll('input[name=filter]'));
	const inputImage = document.getElementById('input-image');
	const inputSaturation = document.getElementById('input-saturation');
	
	const image = await ImageLoader.loadFromUrl('https://images.unsplash.com/photo-1525253013412-55c1a69a5738?ixlib=rb-1.2.1&ixid=eyJhcHBfaWQiOjEyMDd9&auto=format&fit=crop&w=1024&q=90');
	const app = new Application();
	
	app.setImage(image);
	app.render();

	document.getElementById('canvas-container').appendChild(app.canvas);
	
	inputBrightness.addEventListener('input', () => app
		.setBrightness(Number.parseFloat(inputBrightness.value)));
	inputContrast.addEventListener('input', () => app
		.setContrast(Number.parseFloat(inputContrast.value)));
	inputExposure.addEventListener('input', () => app
		.setExposure(Number.parseFloat(inputExposure.value)));
	inputSaturation.addEventListener('input', () => app
		.setSaturation(Number.parseFloat(inputSaturation.value)));

	inputImage.addEventListener('change', async (event) => {
		if (event.target.files.length > 0) {
			const image = await ImageLoader.loadFromFile(event.target.files[0]);

			app.setImage(image);
		}
	});
	
	window.addEventListener('change', (event) => {
		if (!inputFilterList.includes(event.target)) {
			return;
		}
		
		app.setFilter(event.target.value);
	});
})();